// js/app.js — INDO HIDEOUT (fix PJAX + idempotent binds)
(function () {
  // ==========================
  // CONFIG: Daftar lagu
  // ==========================
  const tracks = [
    { title: 'Flow',             artist: 'Hideout', src: 'audio/flow - hideout.mp3',              cover: 'audio/covers/cover1.png' },
    { title: 'Hati',             artist: 'Hideout', src: 'audio/Hati di Hideout.mp3',             cover: 'audio/covers/cover1.png' },
    { title: 'Kita Bernyanyi',   artist: 'Hideout', src: 'audio/Kita Bernyanyi - Hideout.mp3',    cover: 'audio/covers/cover1.png' },
    { title: 'Langkah',          artist: 'Hideout', src: 'audio/Langkah Di Hideout.mp3',          cover: 'audio/covers/cover1.png' },
    { title: 'Selamanya',        artist: 'Hideout', src: 'audio/Selamanya - Hideout.mp3',         cover: 'audio/covers/cover1.png' },
    { title: 'Senandung',        artist: 'Hideout', src: 'audio/Senandung Hideout.mp3',           cover: 'audio/covers/cover1.png' },
    { title: 'Senja',            artist: 'Hideout', src: 'audio/Senja di Hideout.mp3',            cover: 'audio/covers/cover1.png' },
    { title: 'Suasana',          artist: 'Hideout', src: 'audio/Suasana Hideout.mp3',             cover: 'audio/covers/cover1.png' },
    { title: 'Tempat Kita',      artist: 'Hideout', src: 'audio/Tempat Kita (Hideout).mp3',       cover: 'audio/covers/cover1.png' },
  ];

  // ==========================
  // EL / State
  // ==========================
  const KEY = 'ih_player_v1';

  const navLinks = () => Array.from(document.querySelectorAll('.nav a'));
  const mainEl   = () => document.querySelector('#pjax-root') || document.querySelector('main'); // target pjax replace

  // Player UI (elemen ini ada di luar <main> agar tidak ke-replace)
  const musicBtn   = document.getElementById('musicBtn');
  const musicModal = document.getElementById('musicModal');
  const musicClose = document.getElementById('musicClose');
  const playlist   = document.getElementById('playlist');
  const audio      = document.getElementById('audio');
  const nowTitle   = document.getElementById('nowTitle');
  const nowArtist  = document.getElementById('nowArtist');
  const nowCover   = document.getElementById('nowCover');

  // ==========================
  // Helpers: Loading bar
  // ==========================
  function startLoading(linkEl) {
    document.body.classList.add('nav-loading');
    linkEl?.classList.add('is-loading');
  }
  function stopLoading() {
    document.body.classList.remove('nav-loading');
    document.querySelectorAll('.nav a.is-loading').forEach(a => a.classList.remove('is-loading'));
  }

  // ==========================
  // Playlist / Player
  // ==========================
  function renderList() {
    if (!playlist) return;
    playlist.innerHTML = '';
    tracks.forEach((t, i) => {
      const li = document.createElement('li');
      li.className = 'track';
      li.innerHTML = `
        <img class="track__cover" src="${t.cover || ''}" alt="" />
        <div class="track__meta">
          <p class="track__title" title="${t.title}">${t.title}</p>
          <p class="track__artist">${t.artist}</p>
        </div>
        <button class="icon-btn" data-act="info" title="Info">ℹ️</button>
        <button class="icon-btn" data-act="play" title="Play" aria-label="Play ${t.title}">▶</button>
      `;
      li.addEventListener('click', (e) => {
        const btn = e.target.closest('button');
        if (!btn) return;
        const act = btn.dataset.act;
        if (act === 'play') {
          playTrack(i);
        } else if (act === 'info') {
          alert(`${t.title} — ${t.artist}`);
        }
      });
      playlist.appendChild(li);
    });
  }

  function updateNow(t) {
    if (nowTitle)  nowTitle.textContent  = t?.title || 'Now Playing';
    if (nowArtist) nowArtist.textContent = t?.artist || '';
    if (nowCover)  nowCover.src          = t?.cover || '';
    if (nowCover)  nowCover.alt          = t?.title || '';
  }

  function safeSrc(path) {
    // Amanin path yang ada spasi; jangan double-encode
    return /%[0-9A-Fa-f]{2}/.test(path) ? path : encodeURI(path);
  }

  function playTrack(i) {
    const t = tracks[i];
    if (!t || !audio) return;
    updateNow(t);
    audio.src = safeSrc(t.src);
    audio.play().then(() => {
      saveState({ src: audio.src, paused: false, title: t.title, artist: t.artist, cover: t.cover, currentTime: Math.floor(audio.currentTime || 0) });
    }).catch(() => {/* autoplay mungkin diblokir */});
  }

  function saveState(extra = {}) {
    if (!audio) return;
    const state = {
      src: audio.currentSrc || audio.src || '',
      currentTime: Math.floor(audio.currentTime || 0),
      paused: audio.paused,
      title: nowTitle?.textContent || '',
      artist: nowArtist?.textContent || '',
      cover: nowCover?.src || '',
      ...extra
    };
    try { localStorage.setItem(KEY, JSON.stringify(state)); } catch (e) {}
  }

  function restoreState() {
    let state = {};
    try { state = JSON.parse(localStorage.getItem(KEY) || '{}'); } catch (e) {}
    if (!state || !state.src || !audio) return;

    updateNow(state);
    audio.src = state.src;

    audio.addEventListener('loadedmetadata', () => {
      if (state.currentTime) audio.currentTime = state.currentTime;
      if (!state.paused) {
        audio.play().catch(() => {
          const n = document.createElement('div');
          n.textContent = '▶ Tap untuk lanjutkan musik';
          Object.assign(n.style, { position:'fixed', left:'50%', bottom:'20px', transform:'translateX(-50%)',
            background:'rgba(0,0,0,.75)', color:'#fff', padding:'8px 12px', borderRadius:'10px', zIndex:9999 });
          document.body.appendChild(n);
          const resume = () => { audio.play().finally(() => n.remove()); document.removeEventListener('pointerdown', resume); };
          document.addEventListener('pointerdown', resume, { once: true });
        });
      }
    }, { once: true });
  }

  // Persist periodically
  audio?.addEventListener('timeupdate', () => {
    if ((audio.currentTime | 0) % 5 === 0) saveState();
  });
  audio?.addEventListener('pause', () => saveState());
  audio?.addEventListener('play',  () => saveState({ paused: false }));

  // Modal open/close (bind sekali)
  if (musicBtn && !musicBtn.dataset.bound) {
    const openModal  = () => musicModal?.classList.add('open');
    const closeModal = () => musicModal?.classList.remove('open');
    musicBtn.addEventListener('click', openModal);
    musicClose?.addEventListener('click', closeModal);
    musicModal?.addEventListener('click', (e) => { if (e.target === musicModal) closeModal(); });
    document.addEventListener('keydown', (e) => { if (e.key === 'Escape') closeModal(); });
    musicBtn.dataset.bound = '1';
  }

  // ==========================
  // PJAX ringan (replace <main>)
  // ==========================
  function updateActiveNav(pathname) {
    navLinks().forEach(a => {
      a.removeAttribute('aria-current');
      const url = new URL(a.href, location.href);
      if (url.pathname === pathname) a.setAttribute('aria-current', 'page');
    });
  }

  function pjaxClickHandler(e) {
    const a = e.target.closest('a');
    if (!a) return;
    if (!a.closest('.nav')) return; // hanya intercept topbar
    const href = a.getAttribute('href');
    if (!href || href.startsWith('#') || href.startsWith('mailto:') || href.startsWith('tel:') || href.startsWith('roblox://')) return;
    const url = new URL(href, location.href);
    if (url.origin !== location.origin) return;
    if (a.target && a.target !== '_self') return;
    if (e.metaKey || e.ctrlKey || e.shiftKey || e.button !== 0) return;

    e.preventDefault();
    startLoading(a);

    fetch(url.href, { headers: { 'X-PJAX': 'true' } })
      .then(r => r.text())
      .then(html => {
        const parser = new DOMParser();
        const doc = parser.parseFromString(html, 'text/html');
        const newMain = doc.querySelector('#pjax-root') || doc.querySelector('main');
        if (!newMain || !mainEl()) { location.href = url.href; return; }

        // Tutup overlay dari halaman lama kalau ada
        document.querySelectorAll('.lightbox.open').forEach(x => x.classList.remove('open'));

        // Ganti konten utama
        mainEl().replaceWith(newMain);
        document.title = doc.title || document.title;
        history.pushState({ url: url.href }, '', url.href);

        updateActiveNav(url.pathname);
        hydratePage(); // re-bind fitur halaman

        window.scrollTo(0, 0);
        stopLoading();
      })
      .catch(() => { location.href = url.href; });
  }

  function setupPjax() {
    // pastikan handler tidak terpasang dobel
    document.removeEventListener('click', pjaxClickHandler);
    document.addEventListener('click', pjaxClickHandler);

    window.addEventListener('popstate', () => {
      startLoading();
      fetch(location.href, { headers: { 'X-PJAX': 'true' } })
        .then(r => r.text())
        .then(html => {
          const parser = new DOMParser();
          const doc = parser.parseFromString(html, 'text/html');
          const newMain = doc.querySelector('#pjax-root') || doc.querySelector('main');
          if (newMain && mainEl()) {
            document.querySelectorAll('.lightbox.open').forEach(x => x.classList.remove('open'));
            mainEl().replaceWith(newMain);
            document.title = doc.title || document.title;
            updateActiveNav(location.pathname);
            hydratePage();
            window.scrollTo(0, 0);
          }
          stopLoading();
        })
        .catch(() => stopLoading());
    });
  }

  // ==========================
  // Hydrate fitur halaman (dipanggil awal & setiap selesai PJAX)
  // ==========================
  function hydratePage() {
    // --- Hamburger: bind SEKALI ---
    const hamb = document.getElementById('hamb');
    const menu = document.getElementById('menu');
    if (hamb && !hamb.dataset.bound) {
      hamb.addEventListener('click', () => {
        const open = menu.style.display === 'block';
        menu.style.display = open ? '' : 'block';
        hamb.setAttribute('aria-expanded', String(!open));
      });
      hamb.dataset.bound = '1';
    }

    // Footer tahun
    const y = document.getElementById('year');
    if (y) y.textContent = new Date().getFullYear();

    // --- Gallery Lightbox: hindari dobel handler ---
    const grid = document.getElementById('galleryGrid');
    const lb   = document.getElementById('lightbox');
    const lbImg= document.getElementById('lightboxImg');

    if (window.__lbEsc) {
      document.removeEventListener('keydown', window.__lbEsc);
      window.__lbEsc = null;
    }
    if (grid && lb && lbImg) {
      const newGrid = grid.cloneNode(true);
      grid.parentNode.replaceChild(newGrid, grid);

      newGrid.addEventListener('click', (e) => {
        const t = e.target;
        if (t.tagName === 'IMG') {
          lbImg.src = t.src.replace(/w=\d+/, 'w=1600');
          lb.classList.add('open');
        }
      });
      lb.addEventListener('click', () => lb.classList.remove('open'));

      window.__lbEsc = (e) => { if (e.key === 'Escape') lb.classList.remove('open'); };
      document.addEventListener('keydown', window.__lbEsc);
    }

    // --- Contact form → /contact.php (bind sekali) ---
    const form = document.getElementById('contactForm');
    const help = document.getElementById('formHelp');
    const btn  = document.getElementById('sendBtn');
    if (form && !form.dataset.bound) {
      form.dataset.bound = '1';
      const isValidEmail = v => /^[^@\s]+@[^@\s]+\.[^@\s]+$/.test(v);
      form.addEventListener('submit', async (e) => {
        e.preventDefault();
        const name    = document.getElementById('name').value.trim();
        const email   = document.getElementById('email').value.trim();
        const message = (document.getElementById('message')?.value || '').trim();
        const honeypot= (document.getElementById('website')?.value || '').trim();
        if (honeypot) { help.textContent = 'Terima kasih.'; return; }
        if (!name || !email) { help.textContent = 'Nama dan email wajib diisi.'; return; }
        if (!isValidEmail(email)) { help.textContent = 'Format email tidak valid.'; return; }

        help.textContent = 'Mengirim…'; btn && (btn.disabled = true);
        try {
          const res = await fetch('/contact.php', {
            method: 'POST', headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ name, email, message, page: location.href, ua: navigator.userAgent })
          });
          const data = await res.json().catch(() => ({}));
          if (!res.ok || !data.ok) throw new Error(data.msg || 'Gagal mengirim');
          help.textContent = 'Terima kasih! Pesanmu sudah terkirim.'; form.reset();
        } catch {
          help.textContent = 'Gagal mengirim. Coba lagi sebentar.';
        } finally { btn && (btn.disabled = false); }
      });
    }

    // --- Sponsorship filter (bind sekali per render) ---
    const sponsorGrid = document.getElementById('sponsorGrid');
    const tierBtns = document.querySelectorAll('button[data-tier]');
    if (sponsorGrid && tierBtns.length && !sponsorGrid.dataset.bound) {
      sponsorGrid.dataset.bound = '1';
      tierBtns.forEach(btn => {
        btn.addEventListener('click', () => {
          const tier = btn.dataset.tier;
          sponsorGrid.querySelectorAll('.card').forEach(c => {
            c.style.display = (tier === 'all' || c.dataset.tier === tier) ? '' : 'none';
          });
        });
      });
    }
  }

  // ==========================
  // INIT
  // ==========================
  renderList();
  restoreState();
  hydratePage();
  setupPjax();

  // Simpan state saat mau pergi/hidden (fallback)
  window.addEventListener('beforeunload', () => saveState());
  document.addEventListener('visibilitychange', () => { if (document.hidden) saveState(); });
})();
